/**
*****************************************************************************************
*     Copyright(c) 2017, Realtek Semiconductor Corporation. All rights reserved.
*****************************************************************************************
  * @file     bas.h
  * @brief    Head file for using battery service.
  * @details  BAS data structs and external functions declaration.
  * @author
  * @date
  * @version  v1.0
  * *************************************************************************************
  */

/* Define to prevent recursive inclusion */
#ifndef _BAS_H_
#define _BAS_H_

#ifdef  __cplusplus
extern "C" {
#endif      /* __cplusplus */

/* Add Includes here */
#include "profile_server.h"


/** @defgroup BAS Battery Service
  * @brief  Battery service
   * @details

     The Battery Service exposes the state of a battery within a device.

     Battery Service generally makes up a profile with some other services, and it can provide the state of a battery within a device.

     The default supported feature provided by BAS in RTL8763B SDK is the notify property of battery level characteristic,
     and the application developers can modify the feature supported by BAS according to their own requirements.
     The specific configuration process can be achieved by modifying file @ref bas_config.h.

    * <b>Example usage</b>
    * \code{.c}

         #define BAS_BATTERY_LEVEL_NOTIFY_SUPPORT    1

    * \endcode

     Application shall register battery service when initialization through @ref bas_add_service function.

     Application can set the battery level of BAS through @ref bas_set_parameter function.

     Application can send the battery level value of BAS to the client with a notification through @ref bas_battery_level_value_notify function.

  * @{
  */

/*============================================================================*
 *                         Macros
 *============================================================================*/
/** @defgroup BAS_Exported_Macros BAS Exported Macros
  * @brief
  * @{
  */

#define BAS_READ_BATTERY_LEVEL  1
#define BAS_NOTIFY_BATTERY_LEVEL_ENABLE  1
#define BAS_NOTIFY_BATTERY_LEVEL_DISABLE  2

/** End of BAS_Exported_Macros
* @}
*/

/*============================================================================*
 *                         Types
 *============================================================================*/
/** @defgroup BAS_Exported_Types BAS Exported Types
  * @brief
  * @{
  */

/**
*  @brief Battery service parameter type
*/
typedef enum
{
    BAS_PARAM_BATTERY_LEVEL = 0x01,
} T_BAS_PARAM_TYPE;

/**
*  @brief set battery service parameter upstream message data
*/
typedef union
{
    uint8_t notification_indification_index;
    uint8_t read_value_index;
} T_BAS_UPSTREAM_MSG_DATA;

/**
*  @brief set battery service parameter upstream callback data
*/
typedef struct
{
    uint8_t                 conn_id;
    T_SERVICE_CALLBACK_TYPE msg_type;
    T_BAS_UPSTREAM_MSG_DATA msg_data;
} T_BAS_CALLBACK_DATA;

/** End of BAS_Exported_Types
* @}
*/


/*============================================================================*
 *                         Functions
 *============================================================================*/
/** @defgroup BAS_Exported_Functions BAS Exported Functions
  * @brief
  * @{
  */

/**
  * @brief       Add battery service to the BLE stack database.
  *
  *
  * @param[in]   p_func  Callback when service attribute was read, write or cccd update.
  * @return Service id generated by the BLE stack: @ref T_SERVER_ID.
  * @retval 0xFF Operation failure.
  * @retval Others Service id assigned by stack.
  *
  * <b>Example usage</b>
  * \code{.c}
     void profile_init()
     {
         server_init(1);
         bas_id = bas_add_service(app_handle_profile_message);
     }
  * \endcode
  */
T_SERVER_ID bas_add_service(void *p_func);

/**
 * @brief       Set a battery service parameter.
 *
 *              NOTE: You can call this function with a battery service parameter type and it will set the
 *                      battery service parameter.  Battery service parameters are defined in @ref T_BAS_PARAM_TYPE.
 *                      If the "len" field sets to the size of a "uint16_t" ,the
 *                      "p_value" field must point to a data with type of "uint16_t".
 *
 * @param[in]   param_type   Battery service parameter type: @ref T_BAS_PARAM_TYPE
 * @param[in]   length       Length of data to write
 * @param[in]   p_value Pointer to data to write.  This is dependent on
                        the parameter type and will be cast to the appropriate data type.
                        If param_type is set to @ref BAS_PARAM_BATTERY_LEVEL, p_value pointer to the current
                        charge level of a battery. Unit is org.bluetooth.unit.percentage. Minimum value is 0,
                        and maximum value is 100. 100% represents fully charged while 0% represents fully discharged.
 *
 * @return Operation result.
 * @retval true Operation success.
 * @retval false Operation failure.
 *
 * <b>Example usage</b>
 * \code{.c}
    void test(void)
    {
        uint8_t battery_level = 80;
        bas_set_parameter(BAS_PARAM_BATTERY_LEVEL, 1, &battery_level);
    }
 * \endcode
 */
bool bas_set_parameter(T_BAS_PARAM_TYPE param_type, uint8_t length, uint8_t *p_value);


/**
 * @brief       Send notify battery level notification data .
 *
 *
 * @param[in]   conn_id        Connection id.
 * @param[in]   service_id     Service id.
 * @param[in]   battery_level  The current charge level of a battery.
                               Unit is org.bluetooth.unit.percentage. Minimum value is 0, and maximum value is 100.
                               100% represents fully charged while 0% represents fully discharged.
 * @return Operation result.
 * @retval true Operation success.
 * @retval false Operation failure.
 *
 * <b>Example usage</b>
 * \code{.c}
    void test(void)
    {
        uint8_t battery_level = 90;
        bas_battery_level_value_notify(conn_id, bas_id, battery_level);
    }
 * \endcode
 */
bool bas_battery_level_value_notify(uint8_t conn_id, T_SERVER_ID service_id, uint8_t battery_level);


/**
 * @brief       Confirm for read battery level value request.
 *
 *
 * @param[in]   conn_id        Connection id.
 * @param[in]   service_id     Service id.
 * @param[in]   battery_level  The current charge level of a battery.
                               Unit is org.bluetooth.unit.percentage. Minimum value is 0, and maximum value is 100.
                               100% represents fully charged while 0% represents fully discharged.

 * @return Operation result.
 * @retval true Operation success.
 * @retval false Operation failure.
 *
 * <b>Example usage</b>
 * \code{.c}
    void test(void)
    {
        uint8_t battery_level = 90;
        bas_battery_level_value_read_confirm(conn_id, bas_id, battery_level);
    }
 * \endcode
 */
bool bas_battery_level_value_read_confirm(uint8_t conn_id, T_SERVER_ID service_id,
                                          uint8_t battery_level);

/** @} End of BAS_Exported_Functions */

/** @} End of BAS */


#ifdef  __cplusplus
}
#endif      /*  __cplusplus */

#endif  /* _BAS_H_ */
